<?php
// delete_folder.php
include "db.php";
session_start();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Method not allowed');
}

$folder_id = isset($_POST['folder_id']) ? intval($_POST['folder_id']) : 0;
$recursive = isset($_POST['recursive']) && $_POST['recursive'] === '1' ? true : false;
$parent_redirect = isset($_POST['parent_id']) && $_POST['parent_id'] !== '' ? intval($_POST['parent_id']) : null;

if ($folder_id <= 0) {
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Invalid folder id'];
    $loc = 'drive.php' . ($parent_redirect !== null ? '?folder=' . $parent_redirect : '');
    header('Location: ' . $loc);
    exit;
}

// Check folder exists
$chk = $conn->prepare("SELECT id, name FROM folders WHERE id = ?");
$chk->bind_param("i", $folder_id);
$chk->execute();
$folder = $chk->get_result()->fetch_assoc();

if (!$folder) {
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Folder not found'];
    $loc = 'drive.php' . ($parent_redirect !== null ? '?folder=' . $parent_redirect : '');
    header('Location: ' . $loc);
    exit;
}

/**
 * Helper: get all descendant folder ids (breadth-first)
 */
function get_descendant_folder_ids($conn, $startId) {
    $result = [];
    $queue = [$startId];
    while (!empty($queue)) {
        $id = array_shift($queue);
        $result[] = $id;
        $stmt = $conn->prepare("SELECT id FROM folders WHERE parent_id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $res = $stmt->get_result();
        while ($r = $res->fetch_assoc()) {
            $queue[] = (int)$r['id'];
        }
    }
    return $result;
}

if (!$recursive) {
    // Non-recursive: ensure folder is empty (no files, no child folders)
    $fchk = $conn->prepare("SELECT COUNT(*) AS cnt FROM folders WHERE parent_id = ?");
    $fchk->bind_param("i", $folder_id);
    $fchk->execute();
    $hasSub = $fchk->get_result()->fetch_assoc()['cnt'];

    $filechk = $conn->prepare("SELECT COUNT(*) AS cnt FROM files WHERE folder_id = ?");
    $filechk->bind_param("i", $folder_id);
    $filechk->execute();
    $hasFiles = $filechk->get_result()->fetch_assoc()['cnt'];

    if ($hasSub > 0 || $hasFiles > 0) {
        $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Folder is not empty. Use recursive delete to remove everything.'];
        $loc = 'drive.php?folder=' . $folder_id;
        header('Location: ' . $loc);
        exit;
    }

    // Safe to delete single folder
    $d = $conn->prepare("DELETE FROM folders WHERE id = ?");
    $d->bind_param("i", $folder_id);
    if ($d->execute()) {
        $_SESSION['upload_flash'] = ['status' => 'success', 'msg' => 'Folder deleted: ' . $folder['name']];
    } else {
        $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Failed to delete folder'];
    }
    $loc = 'drive.php' . ($parent_redirect !== null ? '?folder=' . $parent_redirect : '');
    header('Location: ' . $loc);
    exit;
}

// ---------- Recursive delete ----------
$ids = get_descendant_folder_ids($conn, $folder_id); // includes root

// Begin transaction
$conn->begin_transaction();

try {
    // 1) Select files that belong to these folders (folder_id IN (...))
    $placeholders = implode(',', array_fill(0, count($ids), '?'));
    $types = str_repeat('i', count($ids));
    $params = $ids;

    // prepare dynamic statement to fetch files
    $sqlFetchFiles = "SELECT id, stored_name, original_name FROM files WHERE folder_id IN ($placeholders)";
    $stmtFiles = $conn->prepare($sqlFetchFiles);
    // bind params dynamically
    $stmtFiles->bind_param($types, ...$params);
    $stmtFiles->execute();
    $filesRes = $stmtFiles->get_result();

    $filesToDelete = [];
    while ($f = $filesRes->fetch_assoc()) {
        $filesToDelete[] = $f;
    }

    // 2) Delete file records
    $sqlDeleteFiles = "DELETE FROM files WHERE folder_id IN ($placeholders)";
    $delFilesStmt = $conn->prepare($sqlDeleteFiles);
    $delFilesStmt->bind_param($types, ...$params);
    $delFilesStmt->execute();

    // 3) Delete folder records (all collected ids)
    $sqlDeleteFolders = "DELETE FROM folders WHERE id IN ($placeholders)";
    $delFoldersStmt = $conn->prepare($sqlDeleteFolders);
    $delFoldersStmt->bind_param($types, ...$params);
    $delFoldersStmt->execute();

    // Commit DB changes
    $conn->commit();

    // 4) Remove files from filesystem (do this after commit)
    foreach ($filesToDelete as $f) {
        $p = __DIR__ . '/uploads/' . $f['stored_name'];
        if (is_file($p)) @unlink($p);
    }

    $_SESSION['upload_flash'] = ['status' => 'success', 'msg' => 'Folder and contents deleted: ' . $folder['name']];
    $loc = 'drive.php' . ($parent_redirect !== null ? '?folder=' . $parent_redirect : '');
    header('Location: ' . $loc);
    exit;

} catch (Exception $e) {
    $conn->rollback();
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Failed to delete folder: ' . $e->getMessage()];
    $loc = 'drive.php' . ($parent_redirect !== null ? '?folder=' . $parent_redirect : '');
    header('Location: ' . $loc);
    exit;
}
