<?php
// download.php — polished file page with inline preview (fixed)
include "db.php";

if (!isset($_GET["token"]) || trim($_GET["token"]) === "") {
    http_response_code(400);
    exit("Invalid request.");
}

$token = $_GET["token"];
$stmt = $conn->prepare("SELECT * FROM files WHERE token = ?");
$stmt->bind_param("s", $token);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    http_response_code(404);
    exit("File not found or link expired.");
}

$file = $result->fetch_assoc();
$storedName = $file["stored_name"];
$originalName = $file["original_name"];
$filePath = __DIR__ . "/uploads/" . $storedName;
$webPath = dirname($_SERVER['SCRIPT_NAME']);
$webPath = rtrim($webPath, '/\\');
$publicUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http')
           . '://' . $_SERVER['HTTP_HOST'] . $webPath . '/uploads/' . rawurlencode($storedName);

// helper: human readable size
function human_filesize($bytes, $decimals = 2) {
    $sz = ['B','KB','MB','GB','TB'];
    if ($bytes == 0) return '0 B';
    $factor = floor((strlen((string)$bytes) - 1) / 3);
    return sprintf("%.{$decimals}f", $bytes / pow(1024, $factor)) . ' ' . $sz[$factor];
}

$exists = is_file($filePath) && is_readable($filePath);
$size = $exists ? filesize($filePath) : 0;
$uploaded_at = isset($file['uploaded_at']) ? $file['uploaded_at'] : '';
$ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));

// detect mime if possible
$mime = 'application/octet-stream';
if ($exists && function_exists('finfo_open')) {
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    if ($finfo) {
        $det = finfo_file($finfo, $filePath);
        if ($det) $mime = $det;
        finfo_close($finfo);
    }
}

// a small helper to decide previewability
$imageExt = ['jpg','jpeg','png','gif','webp','bmp','svg'];
$plainExt = ['txt','csv','log','md','json','xml','php','js','css'];
$canPreviewInline = $exists && (
    in_array($ext, $imageExt) ||
    $ext === 'pdf' ||
    in_array($ext, $plainExt)
);

// Google Viewer fallback for Office docs
$officeExt = ['doc','docx','xls','xlsx','ppt','pptx','odt','ods'];
$useGoogleViewer = in_array($ext, $officeExt);

// Serve download through PHP if dl=1 (safer if you want access control later)
if (isset($_GET['dl']) && $_GET['dl'] === '1') {
    if (!$exists) {
        http_response_code(404);
        exit('File not found.');
    }
    // Send headers and stream file
    header('Content-Description: File Transfer');
    header('Content-Type: ' . ($mime ?: 'application/octet-stream'));
    header('Content-Disposition: attachment; filename="' . basename($originalName) . '"');
    header('Content-Transfer-Encoding: binary');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . $size);
    // Clear output buffer
    while (ob_get_level()) ob_end_clean();
    readfile($filePath);
    exit;
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title><?= htmlspecialchars($originalName) ?> — Shared File</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .file-card { border-left: 4px solid #0d6efd; }
    .meta-key { color: #6c757d; font-size: .9rem; }
    .preview-area { min-height: 220px; background: #fff; border: 1px solid #e9ecef; border-radius: .25rem; overflow:auto; padding: .75rem; }
    .file-icon { font-size: 2.5rem; }
    pre.plain { white-space: pre-wrap; word-break: break-word; font-size: .9rem; }
    @media (max-width: 767px) {
      .preview-area { margin-top: .75rem; }
    }
  </style>
</head>
<body class="bg-light">
  <nav class="navbar navbar-dark bg-primary">
    <div class="container">
      <a class="navbar-brand" href="index.php">Share — Files</a>
      <div class="text-white small">Secure file sharing</div>
    </div>
  </nav>

  <main class="container py-4">
    <div class="row g-4">
      <div class="col-12">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <h4 class="mb-1"><?= htmlspecialchars($originalName) ?></h4>
            <div class="text-muted small">Shared file</div>
          </div>
          <div>
            </a>
            <?php if ($exists): ?>
              <a class="btn btn-success btn-sm" href="?token=<?= rawurlencode($token) ?>&dl=1">
                ⬇ Download
              </a>
            <?php else: ?>
              <span class="badge bg-danger">Missing</span>
            <?php endif; ?>
          </div>
        </div>
      </div>

      <div class="col-lg-4">
        <div class="card file-card shadow-sm">
          <div class="card-body">
            <div class="d-flex align-items-center mb-3">
              <div class="me-3 text-primary file-icon">
                <?php
                  if (in_array($ext, ['pdf'])) echo "📄";
                  elseif (in_array($ext, ['jpg','jpeg','png','gif','webp','bmp','svg'])) echo "🖼️";
                  elseif (in_array($ext, ['zip','rar','7z','tar','gz'])) echo "🗜️";
                  else echo "📎";
                ?>
              </div>
              <div>
                <div class="fw-semibold"><?= htmlspecialchars($originalName) ?></div>
                <div class="text-muted small"><?= htmlspecialchars($mime) ?></div>
              </div>
            </div>

            <dl class="row mb-2">
              <dt class="col-5 meta-key">Size</dt>
              <dd class="col-7"><?= $exists ? human_filesize($size) : '-' ?></dd>

              <dt class="col-5 meta-key">Uploaded</dt>
              <dd class="col-7"><?= $uploaded_at ? htmlspecialchars($uploaded_at) : '-' ?></dd>

              <dt class="col-5 meta-key">Extension</dt>
              <dd class="col-7"><?= htmlspecialchars($ext) ?></dd>

              <dt class="col-5 meta-key">Token</dt>
              <dd class="col-7"><code style="word-break:break-all;"><?= htmlspecialchars($token) ?></code></dd>
            </dl>

            <div class="d-grid gap-2">
              <?php if ($exists): ?>
                <a class="btn btn-outline-primary" href="#" id="previewToggleBtn">Preview</a>
                <a class="btn btn-outline-secondary" href="<?= htmlspecialchars($publicUrl) ?>" target="_blank">Open in new tab</a>
              <?php endif; ?>

              <div class="btn-group" role="group">
                <button class="btn btn-sm btn-light" id="copyLinkBtn" data-link="<?= htmlspecialchars((isset($_SERVER['HTTPS']) && $_SERVER['HTTPS']==='on'?'https':'http') . '://' . $_SERVER['HTTP_HOST'] . $webPath . '/download.php?token=' . rawurlencode($token)) ?>">Copy share link</button>
                <a class="btn btn-sm btn-outline-danger" href="index.php">Close</a>
              </div>
            </div>
          </div>
        </div>

        <?php if ($useGoogleViewer && $exists): ?>
          <div class="mt-3 small text-muted">
            Can't preview this file inline? Try opening in <a href="https://docs.google.com/gview?url=<?= urlencode($publicUrl) ?>&embedded=true" target="_blank">Google Docs Viewer</a>.
          </div>
        <?php endif; ?>
      </div>

      <div class="col-lg-8">
        <div id="previewContainer" class="<?= $canPreviewInline ? '' : 'd-none' ?>">
          <?php if (!$exists): ?>
            <div class="alert alert-danger">File is missing on the server.</div>
          <?php else: ?>

            <?php if (in_array($ext, $imageExt)): ?>
              <div class="preview-area d-flex align-items-center justify-content-center">
                <img src="<?= htmlspecialchars($publicUrl) ?>" alt="<?= htmlspecialchars($originalName) ?>" class="img-fluid" />
              </div>

            <?php elseif ($ext === 'pdf'): ?>
              <div class="preview-area">
                <embed src="<?= htmlspecialchars($publicUrl) ?>" type="application/pdf" width="100%" height="650px">
              </div>

            <?php elseif (in_array($ext, $plainExt)): ?>
              <div class="preview-area">
                <pre class="plain"><?= htmlspecialchars(@file_get_contents($filePath)) ?></pre>
              </div>

            <?php else: ?>
              <div class="alert alert-secondary">
                No inline preview available for this file type.
                <?php if ($useGoogleViewer): ?>
                  &nbsp; You can try <a href="https://docs.google.com/gview?url=<?= urlencode($publicUrl) ?>&embedded=true" target="_blank">Google Docs Viewer</a>.
                <?php endif; ?>
              </div>
            <?php endif; ?>

          <?php endif; ?>
        </div>

        <!-- If preview not supported show a friendly card with suggestion -->
        <div id="noPreviewBox" class="<?= $canPreviewInline ? 'd-none' : '' ?>">
          <div class="card shadow-sm">
            <div class="card-body">
              <h6 class="card-title">Preview not available</h6>
              <p class="card-text small text-muted">
                This file type cannot be previewed inline. Use the download button or open with Google Docs Viewer.
              </p>
              <?php if ($exists): ?>
                <a class="btn btn-outline-primary btn-sm" href="?token=<?= rawurlencode($token) ?>&dl=1">Download</a>
                <?php if ($useGoogleViewer): ?>
                  <a class="btn btn-outline-secondary btn-sm" href="https://docs.google.com/gview?url=<?= urlencode($publicUrl) ?>&embedded=true" target="_blank">Open in Google Viewer</a>
                <?php endif; ?>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>
    </div>
  </main>

  <footer class="text-center small text-muted py-3">
    &copy; <?= date('Y') ?> Share — Files
  </footer>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
  <script>
    (function(){
      var previewBtn = document.getElementById('previewToggleBtn');
      var previewContainer = document.getElementById('previewContainer');
      var noPreviewBox = document.getElementById('noPreviewBox');

      if (previewBtn) {
        previewBtn.addEventListener('click', function(e){
          e.preventDefault();
          // toggle visibility
          if (previewContainer.classList.contains('d-none')) {
            previewContainer.classList.remove('d-none');
            noPreviewBox.classList.add('d-none');
            previewBtn.textContent = 'Hide Preview';
            previewContainer.scrollIntoView({behavior:'smooth'});
          } else {
            previewContainer.classList.add('d-none');
            noPreviewBox.classList.remove('d-none');
            previewBtn.textContent = 'Preview';
          }
        });
      }

      // Copy share link button
      var copyBtn = document.getElementById('copyLinkBtn');
      if (copyBtn) {
        copyBtn.addEventListener('click', function(){
          var link = this.getAttribute('data-link');
          if (!navigator.clipboard) {
            prompt('Copy this link:', link);
            return;
          }
          navigator.clipboard.writeText(link).then(function(){
            // create a small toast-like feedback using template literal (no escaping issues)
            var old = document.getElementById('copy-feedback');
            if (old) old.remove();
            var badge = document.createElement('div');
            badge.id = 'copy-feedback';
            badge.className = 'position-fixed bottom-0 end-0 p-3';
            badge.style.zIndex = 1080;
            badge.innerHTML = `
              <div class="toast align-items-center text-bg-success border-0 show" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                  <div class="toast-body">Link copied to clipboard</div>
                  <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="this.closest('.position-fixed').remove()"></button>
                </div>
              </div>`;
            document.body.appendChild(badge);
            setTimeout(function(){ var el = document.getElementById('copy-feedback'); if (el) el.remove(); }, 3000);
          }, function(){
            alert('Unable to copy link — please copy it manually: ' + link);
          });
        });
      }
    })();
  </script>
</body>
</html>
