<?php
session_start();

// Require login
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
if ($_SESSION['role'] !== 'admin') {
    die("You do not have permission to access this page.");
}

include "db.php";

/* Helpers */
function human_filesize($bytes, $decimals = 2) {
    $sz = ['B','KB','MB','GB','TB'];
    if ($bytes == 0) return '0 B';
    $factor = floor((strlen((string)$bytes) - 1) / 3);
    return sprintf("%.{$decimals}f", $bytes / pow(1024, $factor)) . ' ' . $sz[$factor];
}

/* Current folder (root = null) */
$currentFolder = (isset($_GET['folder']) && $_GET['folder'] !== '') ? intval($_GET['folder']) : null;

/* Handle Create Folder (POST -> same page) */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'create_folder') {
    $folder_name = trim($_POST['folder_name'] ?? '');
    $parent_id = (isset($_POST['parent_id']) && $_POST['parent_id'] !== '') ? intval($_POST['parent_id']) : null;

    if ($folder_name !== '') {
        if ($parent_id === null) {
            $stmt = $conn->prepare("INSERT INTO folders (name, parent_id, created_at) VALUES (?, NULL, NOW())");
            $stmt->bind_param("s", $folder_name);
        } else {
            $stmt = $conn->prepare("INSERT INTO folders (name, parent_id, created_at) VALUES (?, ?, NOW())");
            $stmt->bind_param("si", $folder_name, $parent_id);
        }
        if ($stmt->execute()) {
            $loc = 'drive.php' . ($parent_id ? '?folder=' . $parent_id : '');
            header("Location: $loc");
            exit;
        } else {
            $create_error = "Unable to create folder. Try again.";
        }
    } else {
        $create_error = "Folder name cannot be empty.";
    }
}

/* Fetch current folder info */
$currentFolderData = null;
if ($currentFolder !== null) {
    $pstmt = $conn->prepare("SELECT * FROM folders WHERE id = ?");
    $pstmt->bind_param("i", $currentFolder);
    $pstmt->execute();
    $currentFolderData = $pstmt->get_result()->fetch_assoc();
    if (!$currentFolderData) {
        $currentFolder = null;
        $currentFolderData = null;
    }
}

/* Fetch child folders */
if ($currentFolder === null) {
    $folderQuery = $conn->prepare("SELECT * FROM folders WHERE parent_id IS NULL ORDER BY name ASC");
} else {
    $folderQuery = $conn->prepare("SELECT * FROM folders WHERE parent_id = ? ORDER BY name ASC");
    $folderQuery->bind_param("i", $currentFolder);
}
$folderQuery->execute();
$folders = $folderQuery->get_result();

/* Fetch files */
if ($currentFolder === null) {
    $fileQuery = $conn->prepare("SELECT * FROM files WHERE folder_id IS NULL ORDER BY uploaded_at DESC");
} else {
    $fileQuery = $conn->prepare("SELECT * FROM files WHERE folder_id = ? ORDER BY uploaded_at DESC");
    $fileQuery->bind_param("i", $currentFolder);
}
$fileQuery->execute();
$files = $fileQuery->get_result();

/* Sidebar top-level folders */
$sideStmt = $conn->prepare("SELECT id, name FROM folders WHERE parent_id IS NULL ORDER BY name ASC");
$sideStmt->execute();
$topFolders = $sideStmt->get_result();

/* Breadcrumb parent */
$parentOfCurrent = null;
if ($currentFolderData && isset($currentFolderData['parent_id'])) {
    $parentOfCurrent = $currentFolderData['parent_id'];
}

/* Base share URL */
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>DACL ShareDrive — Explorer</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/remixicon/fonts/remixicon.css" rel="stylesheet">
  <style>
    :root { --brand: #0d6efd; }
    body { background:#f8f9fa; }
    .folder-card { cursor:pointer; transition: transform .08s ease; }
    .folder-card:hover { transform: translateY(-4px); }
    .sidebar { max-height: 80vh; overflow:auto; }
    iframe.preview-frame { width:100%; height:70vh; border:0; }
    @media (max-width: 767px) { iframe.preview-frame { height:50vh; } }

    .folder-delete-btn { font-size:0.95rem; padding:.25rem .45rem; position:relative; z-index:3; pointer-events:auto; }
    .card .stretched-link { z-index:0; position:relative; }

    .action-icon { font-size: 1.05rem; width:36px; height:36px; display:inline-flex; align-items:center; justify-content:center; border-radius:6px; border:1px solid transparent; background:transparent; }
    .action-icon:hover { background:#f4f6f8; border-color:#e9ecef; }

    .mobile-actions-btn { width:36px; height:36px; display:inline-flex; align-items:center; justify-content:center; border-radius:6px; border:1px solid transparent; background:transparent; }

    .brand-logo { width:36px; height:36px; object-fit:contain; margin-right:.6rem; }
    .brand-title { font-weight:600; color:#fff; }

    /* Actions: desktop vs mobile via Bootstrap utility classes in markup */
    /* Folder card link accessibility */
    .card .card-body { z-index:0; }
  </style>
</head>
<body>
  <nav class="navbar navbar-expand-lg" style="background:var(--brand);">
    <div class="container-fluid">
      <a class="d-flex align-items-center text-decoration-none" href="drive.php">
        <img src="logo.png" alt="logo" class="brand-logo">
        <span class="brand-title text-white">DACL ShareDrive</span>
      </a>

      <div class="ms-auto d-flex align-items-center gap-2">
        <!-- SINGLE Upload button — only one in navbar -->
        <form method="post" action="upload.php" enctype="multipart/form-data" class="d-flex align-items-center me-2">
          <input type="hidden" name="folder_id" value="<?= $currentFolder !== null ? $currentFolder : '' ?>">
          <label class="btn btn-light btn-sm mb-0" style="background:#fff; border:1px solid rgba(0,0,0,0.08);">
            <i class="ri-upload-line me-1"></i> Upload
            <input type="file" name="file" onchange="this.form.submit()" hidden>
          </label>
        </form>

        <a class="btn btn-outline-light btn-sm" href="drive.php">Home</a>
      </div>
    </div>
  </nav>

  <div class="container-fluid py-4">
    <div class="row gx-4">
      <!-- SIDEBAR -->
      <aside class="col-md-3 mb-3">
        <div class="card mb-3">
          <div class="card-body d-flex justify-content-between align-items-center">
            <h6 class="mb-0">Folders</h6>
            <button class="btn btn-sm btn-light" data-bs-toggle="modal" data-bs-target="#createFolderModal">New</button>
          </div>
        </div>

        <div class="card sidebar mb-3">
          <div class="list-group list-group-flush">
            <a href="drive.php" class="list-group-item list-group-item-action <?= $currentFolder===null ? 'active' : '' ?>">📁 Root</a>
            <?php while ($tf = $topFolders->fetch_assoc()): ?>
              <a href="drive.php?folder=<?= $tf['id'] ?>" class="list-group-item list-group-item-action <?= ($currentFolder===$tf['id']) ? 'active' : '' ?>">📁 <?= htmlspecialchars($tf['name']) ?></a>
            <?php endwhile; ?>
          </div>
        </div>

        <div class="card">
          <div class="card-body small text-muted">
            <strong>Tips</strong>
            <ul class="mb-0 mt-2">
              <li>Use <strong>New</strong> to create folders.</li>
              <li>Upload into a folder by opening the folder first, then using Upload.</li>
              <li>Tap the three-dot menu on mobile for file actions.</li>
            </ul>
          </div>
        </div>
      </aside>

      <!-- MAIN -->
      <main class="col-md-9">
        <!-- Flash -->
        <?php
        if (!empty($_SESSION['upload_flash'])) {
            $f = $_SESSION['upload_flash'];
            $cls = ($f['status'] === 'success') ? 'alert-success' : 'alert-danger';
            echo '<div class="alert ' . $cls . ' d-flex justify-content-between align-items-center">';
            echo '<div><strong>' . ($f['status'] === 'success' ? 'Success' : 'Error') . ':</strong> ' . htmlspecialchars($f['msg']);
            if (!empty($f['link'])) echo ' &nbsp; <a href="' . htmlspecialchars($f['link']) . '" target="_blank">Open / Share</a>';
            echo '</div><button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>';
            unset($_SESSION['upload_flash']);
        } elseif (isset($_GET['upload']) && $_GET['upload'] === 'success') {
            echo '<div class="alert alert-success">Upload successful.</div>';
        } elseif (isset($_GET['upload']) && $_GET['upload'] === 'error') {
            $m = isset($_GET['msg']) ? htmlspecialchars($_GET['msg']) : 'Upload error';
            echo '<div class="alert alert-danger"><strong>Error:</strong> ' . $m . '</div>';
        }
        ?>

        <div class="d-flex justify-content-between align-items-center mb-3">
          <div>
            <nav style="--bs-breadcrumb-divider: '›';" aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item"><a href="drive.php">Home</a></li>
                <?php
                  if ($currentFolderData) {
                      if ($parentOfCurrent) {
                          $pnameStmt = $conn->prepare("SELECT id, name, parent_id FROM folders WHERE id = ?");
                          $pnameStmt->bind_param("i", $parentOfCurrent);
                          $pnameStmt->execute();
                          $pinfo = $pnameStmt->get_result()->fetch_assoc();
                          if ($pinfo) echo '<li class="breadcrumb-item"><a href="drive.php?folder=' . $pinfo['id'] . '">' . htmlspecialchars($pinfo['name']) . '</a></li>';
                      }
                      echo '<li class="breadcrumb-item active" aria-current="page">' . htmlspecialchars($currentFolderData['name']) . '</li>';
                  }
                ?>
              </ol>
            </nav>
            <?php if ($currentFolderData): ?>
              <div class="text-muted small mt-1"><a href="drive.php<?= $parentOfCurrent ? '?folder=' . $parentOfCurrent : '' ?>">&larr; Back</a></div>
            <?php endif; ?>
          </div>

          <div class="d-flex gap-2 align-items-center">
            <button class="btn btn-outline-secondary btn-sm" id="toggleViewBtn">Toggle View</button>
          </div>
        </div>

        <?php if (!empty($create_error)): ?><div class="alert alert-danger"><?= htmlspecialchars($create_error) ?></div><?php endif; ?>

        <!-- FOLDERS GRID -->
        <div class="row mb-4" id="foldersGrid">
          <?php if ($folders->num_rows === 0): ?><div class="col-12 mb-2 text-muted">No folders in this location.</div><?php endif; ?>

          <?php while ($folder = $folders->fetch_assoc()):
              $flink = 'drive.php?folder=' . $folder['id'];
          ?>
            <div class="col-sm-6 col-md-4 col-lg-3 mb-3">
              <div class="card folder-card h-100 position-relative"
                   data-folder-id="<?= $folder['id'] ?>"
                   data-folder-name="<?= htmlspecialchars($folder['name'], ENT_QUOTES) ?>"
                   data-parent-id="<?= $currentFolder !== null ? $currentFolder : '' ?>"
                   data-folder-link="<?= $flink ?>">
                <a href="<?= $flink ?>" class="text-decoration-none text-dark stretched-link" aria-hidden="true"></a>
                <div class="card-body">
                  <div class="d-flex align-items-center justify-content-between">
                    <div class="d-flex align-items-center">
                      <div style="font-size:1.4rem; margin-right:.6rem;">📁</div>
                      <div>
                        <div class="fw-semibold"><?= htmlspecialchars($folder['name']) ?></div>
                        <div class="small text-muted">Folder</div>
                      </div>
                    </div>
                    <div>
                      <button type="button" class="btn btn-sm btn-outline-danger folder-delete-btn"
                        data-folder-id="<?= $folder['id'] ?>"
                        data-folder-name="<?= htmlspecialchars($folder['name'], ENT_QUOTES) ?>"
                        data-parent-id="<?= $currentFolder !== null ? $currentFolder : '' ?>"
                        title="Delete folder">
                        <i class="ri-delete-bin-line"></i>
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          <?php endwhile; ?>
        </div>

        <!-- FILES TABLE -->
        <div class="card">
          <div class="card-body p-0">
            <table class="table table-hover mb-0 align-middle">
              <thead class="table-light">
                <tr>
                  <th style="width:55%;">Name</th>
                  <th style="width:15%;">Modified</th>
                  <th style="width:10%;">Size</th>
                  <th style="width:20%;">Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php if ($files->num_rows === 0): ?><tr><td colspan="4" class="text-muted small p-3">No files in this folder.</td></tr><?php endif; ?>

                <?php while ($file = $files->fetch_assoc()):
                    $filePath = 'uploads/' . $file['stored_name'];
                    $size = is_file($filePath) ? filesize($filePath) : 0;
                    $shareLink = $baseUrl . '/download.php?token=' . rawurlencode($file['token']);
                ?>
                  <tr>
                    <td>
                      <div class="fw-medium"><?= htmlspecialchars($file['original_name']) ?></div>
                      <div class="small text-muted"><?= htmlspecialchars($file['original_name']) !== '' ? strtolower(pathinfo($file['original_name'], PATHINFO_EXTENSION)) : '' ?></div>
                    </td>
                    <td class="small text-muted"><?= htmlspecialchars($file['uploaded_at']) ?></td>
                    <td class="small text-muted"><?= $size ? human_filesize($size) : '-' ?></td>
                    <td class="file-actions">
                      <!-- Desktop icons only (visible md+) -->
                      <div class="desktop-icons d-none d-md-inline-flex">
                        <button class="btn action-icon" title="Preview" data-token="<?= htmlspecialchars($file['token']) ?>" data-action="preview"><i class="ri-eye-line"></i></button>

                        <a class="btn action-icon" title="Download" href="download.php?token=<?= htmlspecialchars($file['token']) ?>"><i class="ri-download-line"></i></a>

                        <button class="btn action-icon" title="Share link" data-share="<?= htmlspecialchars($shareLink) ?>" data-action="share"><i class="ri-share-forward-line"></i></button>

                        <button class="btn action-icon text-danger" title="Delete" data-token="<?= htmlspecialchars($file['token']) ?>" data-file-name="<?= htmlspecialchars($file['original_name'], ENT_QUOTES) ?>" data-folder-id="<?= $currentFolder !== null ? $currentFolder : '' ?>" data-action="delete-file">
                          <i class="ri-delete-bin-2-line"></i>
                        </button>
                      </div>

                      <!-- Mobile: three-dot only visible on small screens -->
                      <div class="mobile-dropdown dropdown d-inline-block d-md-none">
                        <button class="mobile-actions-btn" data-bs-toggle="dropdown" aria-expanded="false" title="Actions"><i class="ri-more-2-line"></i></button>
                        <ul class="dropdown-menu dropdown-menu-end">
                          <li><a class="dropdown-item preview-item" href="#" data-token="<?= htmlspecialchars($file['token']) ?>"><i class="ri-eye-line me-2"></i>Preview</a></li>
                          <li><a class="dropdown-item" href="download.php?token=<?= htmlspecialchars($file['token']) ?>"><i class="ri-download-line me-2"></i>Download</a></li>
                          <li><a class="dropdown-item share-item" href="#" data-share="<?= htmlspecialchars($shareLink) ?>"><i class="ri-share-forward-line me-2"></i>Share</a></li>
                          <li><hr class="dropdown-divider"></li>
                          <li><a class="dropdown-item delete-item text-danger" href="#" data-token="<?= htmlspecialchars($file['token']) ?>" data-file-name="<?= htmlspecialchars($file['original_name'], ENT_QUOTES) ?>" data-folder-id="<?= $currentFolder !== null ? $currentFolder : '' ?>"><i class="ri-delete-bin-2-line me-2"></i>Delete</a></li>
                        </ul>
                      </div>

                    </td>
                  </tr>
                <?php endwhile; ?>
              </tbody>
            </table>
          </div>
        </div>

      </main>
    </div>
  </div>

  <!-- Modals -->
  <div class="modal fade" id="createFolderModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-sm modal-dialog-centered"><div class="modal-content">
      <form method="post" action="drive.php<?= $currentFolder !== null ? '?folder=' . $currentFolder : '' ?>">
        <input type="hidden" name="action" value="create_folder">
        <input type="hidden" name="parent_id" value="<?= $currentFolder !== null ? $currentFolder : '' ?>">
        <div class="modal-header"><h5 class="modal-title">New Folder</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body"><div class="mb-3"><label class="form-label">Folder name</label><input type="text" name="folder_name" class="form-control" required maxlength="255" autofocus></div></div>
        <div class="modal-footer"><button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-sm btn-primary">Create</button></div>
      </form>
    </div></div>
  </div>

  <div class="modal fade" id="folderDeleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-sm modal-dialog-centered"><div class="modal-content">
      <form method="post" action="delete_folder.php" id="folderDeleteForm">
        <input type="hidden" name="folder_id" id="modal_folder_id" value="">
        <input type="hidden" name="parent_id" id="modal_parent_id" value="">
        <input type="hidden" name="recursive" id="modal_recursive" value="0">
        <div class="modal-header"><h5 class="modal-title">Delete Folder</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body">
          <p id="folderDeleteMsg">Are you sure?</p>
          <div class="form-check"><input class="form-check-input" type="checkbox" value="1" id="confirm_recursive_checkbox"><label class="form-check-label" for="confirm_recursive_checkbox">Delete recursively (remove all subfolders & files)</label></div>
        </div>
        <div class="modal-footer"><button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-sm btn-danger">Delete</button></div>
      </form>
    </div></div>
  </div>

  <div class="modal fade" id="fileDeleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-sm modal-dialog-centered"><div class="modal-content">
      <form method="post" action="delete_file.php" id="fileDeleteForm">
        <input type="hidden" name="token" id="modal_file_token" value="">
        <input type="hidden" name="folder_id" id="modal_file_folder_id" value="">
        <div class="modal-header"><h5 class="modal-title">Delete File</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body"><p id="fileDeleteMsg">Are you sure you want to delete this file?</p><div class="text-muted small">This action cannot be undone.</div></div>
        <div class="modal-footer"><button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-sm btn-danger">Delete</button></div>
      </form>
    </div></div>
  </div>

  <div class="modal fade" id="previewModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-centered"><div class="modal-content">
      <div class="modal-header"><h5 class="modal-title">Preview</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
      <div class="modal-body p-0"><iframe class="preview-frame" src=""></iframe></div>
    </div></div>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
  <script>
    (function(){
      // Preview (desktop icon)
      document.querySelectorAll('button[data-action="preview"]').forEach(function(btn){
        btn.addEventListener('click', function(e){
          e.preventDefault();
          var token = this.getAttribute('data-token') || this.dataset.token;
          var iframe = document.querySelector('.preview-frame');
          iframe.src = 'preview.php?token=' + encodeURIComponent(token);
          new bootstrap.Modal(document.getElementById('previewModal')).show();
        });
      });

      // Preview (mobile dropdown)
      document.querySelectorAll('.preview-item').forEach(function(a){
        a.addEventListener('click', function(e){
          e.preventDefault();
          var token = this.getAttribute('data-token');
          var iframe = document.querySelector('.preview-frame');
          iframe.src = 'preview.php?token=' + encodeURIComponent(token);
          new bootstrap.Modal(document.getElementById('previewModal')).show();
        });
      });

      // Share (desktop & mobile)
      document.querySelectorAll('button[data-action="share"], .share-item').forEach(function(el){
        el.addEventListener('click', function(e){
          e.preventDefault();
          var link = this.getAttribute('data-share') || this.dataset.share;
          if (!link) return;
          if (navigator.clipboard) {
            navigator.clipboard.writeText(link).then(function(){ alert('Share link copied'); }, function(){ prompt('Copy this link:', link); });
          } else { prompt('Copy this link:', link); }
        });
      });

      // Delete file (desktop)
      document.querySelectorAll('button[data-action="delete-file"]').forEach(function(btn){
        btn.addEventListener('click', function(e){
          e.preventDefault(); e.stopPropagation();
          var token = this.getAttribute('data-token');
          var fname = this.getAttribute('data-file-name') || 'this file';
          var folderId = this.getAttribute('data-folder-id') || '';
          document.getElementById('modal_file_token').value = token;
          document.getElementById('modal_file_folder_id').value = folderId;
          document.getElementById('fileDeleteMsg').innerText = 'Delete file "' + fname + '"?';
          new bootstrap.Modal(document.getElementById('fileDeleteModal')).show();
        });
      });

      // Delete file (mobile)
      document.querySelectorAll('.delete-item').forEach(function(a){
        a.addEventListener('click', function(e){
          e.preventDefault(); e.stopPropagation();
          var token = this.getAttribute('data-token');
          var fname = this.getAttribute('data-file-name') || 'this file';
          var folderId = this.getAttribute('data-folder-id') || '';
          document.getElementById('modal_file_token').value = token;
          document.getElementById('modal_file_folder_id').value = folderId;
          document.getElementById('fileDeleteMsg').innerText = 'Delete file "' + fname + '"?';
          new bootstrap.Modal(document.getElementById('fileDeleteModal')).show();
        });
      });

      // Folder delete (icon)
      document.querySelectorAll('.folder-delete-btn').forEach(function(btn){
        btn.addEventListener('click', function(e){
          e.preventDefault(); e.stopPropagation();
          var id = this.getAttribute('data-folder-id');
          var name = this.getAttribute('data-folder-name');
          var parent = this.getAttribute('data-parent-id') || '';
          document.getElementById('modal_folder_id').value = id;
          document.getElementById('modal_parent_id').value = parent;
          document.getElementById('confirm_recursive_checkbox').checked = false;
          document.getElementById('modal_recursive').value = '0';
          document.getElementById('folderDeleteMsg').innerText = 'Delete folder "' + name + '"?';
          new bootstrap.Modal(document.getElementById('folderDeleteModal')).show();
        });
      });

      // Right-click on folder card opens modal
      document.querySelectorAll('.folder-card').forEach(function(card){
        card.addEventListener('contextmenu', function(e){
          if (e.target.closest('.folder-delete-btn')) return;
          e.preventDefault(); e.stopPropagation();
          var id = this.getAttribute('data-folder-id');
          var name = this.getAttribute('data-folder-name');
          var parent = this.getAttribute('data-parent-id') || '';
          document.getElementById('modal_folder_id').value = id;
          document.getElementById('modal_parent_id').value = parent;
          document.getElementById('confirm_recursive_checkbox').checked = false;
          document.getElementById('modal_recursive').value = '0';
          document.getElementById('folderDeleteMsg').innerText = 'Delete folder "' + name + '"?';
          new bootstrap.Modal(document.getElementById('folderDeleteModal')).show();
        });

        // Click card to open folder (unless click on control)
        card.addEventListener('click', function(e){
          if (e.target.closest('button') || e.target.closest('a') || e.target.closest('.dropdown-toggle') || e.target.closest('.mobile-dropdown')) return;
          var link = this.getAttribute('data-folder-link');
          if (link) location.href = link;
        });
      });

      // Recursive checkbox
      var chk = document.getElementById('confirm_recursive_checkbox');
      if (chk) chk.addEventListener('change', function(){ document.getElementById('modal_recursive').value = this.checked ? '1' : '0'; });

      // Toggle view
      var toggleBtn = document.getElementById('toggleViewBtn');
      if (toggleBtn) toggleBtn.addEventListener('click', function(){ var grid = document.getElementById('foldersGrid'); grid.style.display = (grid.style.display === 'none' ? '' : 'none'); });
    })();
  </script>
</body>
</html>
