<?php
// login.php
// Improved login with secure session settings, rate-limiting and admin redirect

// set secure cookie params before starting session
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
           (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');

session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'domain' => '',           // default host
    'secure' => $isHttps,
    'httponly' => true,
    'samesite' => 'Lax'
]);
session_start();

include "db.php";

$error = "";
$submitted_username = '';

// simple login throttling (session-based)
// after 5 failed attempts within 15 minutes block for 15 minutes
$maxAttempts = 5;
$blockMinutes = 15;

if (!isset($_SESSION['login_attempts'])) {
    $_SESSION['login_attempts'] = 0;
    $_SESSION['login_first_attempt_time'] = time();
}

$blocked = false;
if ($_SESSION['login_attempts'] >= $maxAttempts) {
    $elapsed = time() - ($_SESSION['login_first_attempt_time'] ?? time());
    if ($elapsed < ($blockMinutes * 60)) {
        $blocked = true;
        $remaining = intval(($blockMinutes * 60 - $elapsed) / 60) + 1;
        $error = "Too many failed attempts. Try again in approximately {$remaining} minute(s).";
    } else {
        // reset after block window
        $_SESSION['login_attempts'] = 0;
        $_SESSION['login_first_attempt_time'] = time();
        $blocked = false;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$blocked) {
    $submitted_username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';

    if ($submitted_username === '' || $password === '') {
        $error = "Please enter username and password.";
    } else {
        // small progressive delay to slow automated attempts (optional)
        // sleep for a tiny amount proportional to attempts
        $delayUs = min(300000 * $_SESSION['login_attempts'], 700000); // cap at ~0.7s
        if ($delayUs > 0) usleep($delayUs);

        // lookup user
        $stmt = $conn->prepare("SELECT id, username, password, role FROM users WHERE username = ? LIMIT 1");
        $stmt->bind_param("s", $submitted_username);
        $stmt->execute();
        $result = $stmt->get_result();

        // dummy hash to make timing more consistent when user not found
        $dummy_hash = password_hash('invalid_password_for_timing', PASSWORD_DEFAULT);

        if ($result && $result->num_rows === 1) {
            $user = $result->fetch_assoc();
            $hashed = $user['password'];

            if (password_verify($password, $hashed)) {
                // success: regenerate session id and set session vars
                session_regenerate_id(true);
                $_SESSION['user_id'] = (int)$user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['role'] = $user['role'];

                // reset throttling info
                $_SESSION['login_attempts'] = 0;
                $_SESSION['login_first_attempt_time'] = time();

                // redirect to admin UI for admins, otherwise drive
                if (isset($_GET['next']) && preg_match('/^[a-zA-Z0-9_\-\/\.]+$/', $_GET['next'])) {
                    // basic sanitization of next parameter
                    $next = $_GET['next'];
                } else {
                    $next = ($user['role'] === 'admin') ? 'drive.php' : 'drive.php';
                }

                header("Location: {$next}");
                exit;
            } else {
                // password incorrect
                $_SESSION['login_attempts'] += 1;
                if ($_SESSION['login_attempts'] === 1) $_SESSION['login_first_attempt_time'] = time();
                $error = "Invalid username or password.";
            }
        } else {
            // user not found — still run password_verify against dummy to reduce timing difference
            password_verify($password, $dummy_hash);
            $_SESSION['login_attempts'] += 1;
            if ($_SESSION['login_attempts'] === 1) $_SESSION['login_first_attempt_time'] = time();
            $error = "Invalid username or password.";
        }
    }
}

// Optional: if user already logged in redirect
if (!empty($_SESSION['user_id']) && empty($error) && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $home = ($_SESSION['role'] ?? '') === 'admin' ? 'drive.php' : 'drive.php';
    header("Location: {$home}");
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Login - File Manager</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light d-flex align-items-center" style="height:100vh;">

<div class="container">
  <div class="row justify-content-center">
    <div class="col-md-4">
      <div class="card shadow">
        <div class="card-body">
          <h3 class="card-title text-center mb-3">Login</h3>

          <?php if ($error): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
          <?php endif; ?>

          <form method="post" autocomplete="off" novalidate>
            <div class="mb-3">
              <label class="form-label">Username</label>
              <input type="text" name="username" class="form-control" required
                     value="<?= htmlspecialchars($submitted_username) ?>">
            </div>
            <div class="mb-3">
              <label class="form-label">Password</label>
              <input type="password" name="password" class="form-control" required>
            </div>
            <div class="d-grid">
              <button class="btn btn-primary">Login</button>
            </div>
          </form>

          <div class="mt-3 text-muted small">
            <?php if (isset($_SESSION['login_attempts']) && $_SESSION['login_attempts'] > 0): ?>
              Failed attempts: <?= (int)$_SESSION['login_attempts'] ?>.
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

</body>
</html>
