<?php
// upload.php
include "db.php";
session_start();

function upload_error_message($code) {
    $map = [
        UPLOAD_ERR_OK => "There is no error, the file uploaded with success.",
        UPLOAD_ERR_INI_SIZE => "The uploaded file exceeds the upload_max_filesize directive.",
        UPLOAD_ERR_FORM_SIZE => "The uploaded file exceeds the MAX_FILE_SIZE directive.",
        UPLOAD_ERR_PARTIAL => "The uploaded file was only partially uploaded.",
        UPLOAD_ERR_NO_FILE => "No file was uploaded.",
        UPLOAD_ERR_NO_TMP_DIR => "Missing a temporary folder.",
        UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk.",
        UPLOAD_ERR_EXTENSION => "A PHP extension stopped the file upload.",
    ];
    return $map[$code] ?? "Unknown upload error ({$code}).";
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('HTTP/1.1 405 Method Not Allowed');
    exit('Method not allowed');
}

// Make sure file field exists
if (!isset($_FILES['file'])) {
    // set flash and redirect home
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'No file received'];
    header('Location: drive.php');
    exit;
}

/**
 * Interpret folder_id:
 * - posted empty string -> NULL (root)
 * - numeric string -> integer folder id
 */
$folder_raw = $_POST['folder_id'] ?? '';
$folder_id = ($folder_raw === '') ? null : intval($folder_raw);

/** --- handle upload --- */
$file = $_FILES['file'];

if ($file['error'] !== UPLOAD_ERR_OK) {
    $msg = upload_error_message($file['error']);
    // set flash and redirect back to folder view
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => $msg];
    $loc = 'drive.php' . ($folder_id !== null ? '?folder=' . $folder_id : '');
    header('Location: ' . $loc);
    exit;
}

// sanitize original name and build stored name
$originalName = basename($file['name']);
$originalName = preg_replace('/[\x00-\x1F\x7F]+/u', '', $originalName); // remove control chars
$safeOriginal = preg_replace('/[^A-Za-z0-9\-\._ ]/', '_', $originalName);

// unique stored filename
$storedName = bin2hex(random_bytes(8)) . '_' . $safeOriginal;

// ensure uploads dir exists
$uploadDir = __DIR__ . '/uploads/';
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Unable to create uploads directory'];
        $loc = 'drive.php' . ($folder_id !== null ? '?folder=' . $folder_id : '');
        header('Location: ' . $loc);
        exit;
    }
}

$targetPath = $uploadDir . $storedName;

// move uploaded file
if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => 'Failed to move uploaded file'];
    $loc = 'drive.php' . ($folder_id !== null ? '?folder=' . $folder_id : '');
    header('Location: ' . $loc);
    exit;
}

// generate share token
$token = bin2hex(random_bytes(16));

// insert into DB with correct handling of NULL folder_id
if ($folder_id === null) {
    // insert with literal NULL for folder_id
    $stmt = $conn->prepare("INSERT INTO files (original_name, stored_name, token, uploaded_at, folder_id) VALUES (?, ?, ?, NOW(), NULL)");
    $stmt->bind_param("sss", $originalName, $storedName, $token);
} else {
    // insert with folder_id parameter
    $stmt = $conn->prepare("INSERT INTO files (original_name, stored_name, token, uploaded_at, folder_id) VALUES (?, ?, ?, NOW(), ?)");
    $stmt->bind_param("sssi", $originalName, $storedName, $token, $folder_id);
}

if (!$stmt->execute()) {
    // remove the file if DB insert failed
    if (is_file($targetPath)) unlink($targetPath);
    $err = $conn->error ?: 'DB insert failed';
    $_SESSION['upload_flash'] = ['status' => 'error', 'msg' => $err];
    $loc = 'drive.php' . ($folder_id !== null ? '?folder=' . $folder_id : '');
    header('Location: ' . $loc);
    exit;
}

// success -> set a friendly flash message (and keep the share link in the flash)
$shareLink = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http")
          . "://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['SCRIPT_NAME']) . "/download.php?token=" . $token;

// store message and link
$_SESSION['upload_flash'] = [
    'status' => 'success',
    'msg'    => 'File uploaded successfully: ' . $originalName,
    'link'   => $shareLink
];

// Redirect back to the folder view (PRG)
$loc = 'drive.php';
if ($folder_id !== null) $loc .= '?folder=' . $folder_id;
// Keep GET fallback compatibility but prefer session flash
header('Location: ' . $loc);
exit;
